/* $Id: debugio.c,v 1.16 1998/07/20 21:05:57 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Chris Sutton, updated by Eric Backus */

/* From Toolkit hwinstall hwinst3.c,  Dec19, 1994 */

#include "sema.h"

/* CVR values */
#define E1432_CVR_NEW_X_READ    0x8012
#define E1432_CVR_NEW_Y_READ    0x8013

static char errInfo[80];

#define CHK(func) error = (func); \
    if (error) return error;

/*
 *********************************************************************
 This function wait for a bit to be set/cleared in 96000 ICS reg.
 *********************************************************************
*/
static SHORTSIZ16
waitIcsBit(E1432_MODULE_LIST_NODE *mn, unsigned long bit, long set)
{
    SHORTSIZ16 error;
    long    reg;
    int     j;

#define E1432_ICS_WAIT_MAX 120

    for (j = 0; j < E1432_ICS_WAIT_MAX; j++)
    {
	CHK(i1432_direct_read32_register(mn, E1432_HOSTB_ICS_REG, &reg));
	reg &= bit;
	if (reg && set) break;
	if (!reg && !set) break;
    }

    if (j == E1432_ICS_WAIT_MAX)
    {
	(void) sprintf(errInfo, "Host port ICS bit 0x%X not ready\n", bit);
	i1432_error_info = errInfo;
	return i1432_la_print_error(mn->la, ERR1432_IO);
    }
    return 0;
}

/*
 *********************************************************************
 This function writes a word to an address in the 96000 memory space.
 Since the entire 96000 address space is not mapped to VXI, the
 96000 does the memory access via the host port.  So, the 96000
 must be up and running.

 >busBY     0 for 96000 X mem and A bus, 1 for Y mem B bus
 >dspAddr       96000 address
 >dspData   32-bit data to write
 returns    0 of ok
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_write_dsp_mem(E1432ID hw,
            SHORTSIZ16 chanID,
            SHORTSIZ16 busBY,
            LONGSIZ32 dspAddr,
            LONGSIZ32 dspData)
{
#ifndef HAVE_VTL
    SHORTSIZ16 error;
    E1432_MODULE_LIST_NODE dmn; /* default module list node */
    E1432_MODULE_LIST_NODE *mn;
    unsigned long mem;      /* vxi address of 96000 port */
    char    str[E1432_SICL_NAME_MAX + 2];	/* Allow for ",8" */

    TRACE_PRINTF(0, ("e1432_write_dsp_mem(0x%p, %d, %d, 0x%lx, 0x%lx)\n",
             hw, chanID, busBY, dspAddr, dspData));

    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
    {
        /* allow peek/poke to work before assing_channel..  */
	(void) sprintf(str, "%s,8", i1432_sicl_name);
	dmn.sicl_id = iopen(str);
	if (dmn.sicl_id == 0)
	    return i1432_la_print_error(8, ERR1432_UNABLE_TO_OPEN_SICL);

	dmn.a24_base = (LONGSIZ32 *) (void *)
	imap(dmn.sicl_id, I_MAP_EXTEND, 0, 8, NULL);
	dmn.la = 8;
	dmn.chan_id = 1;
	dmn.nchan = 1;
	mn = &dmn;
    }

    /* set VXI addresses */
    if (busBY)
        mem = E1432_HOSTB_Y_WRITE_REG;
    else
        mem = E1432_HOSTB_X_WRITE_REG;

    /* write address to Memory Interrupt */
    CHK(waitIcsBit(mn, E1432_HOSTB_ICS_TXDE, 1));
    CHK(i1432_direct_write32_register(mn, mem, dspAddr));

    /* write data to Memory Interrupt */
    CHK(waitIcsBit(mn, E1432_HOSTB_ICS_TXDE, 1));
    CHK(i1432_direct_write32_register(mn, mem, dspData));

#endif
    return 0;
}

/*
 *********************************************************************
 Returns -1 for hard failure.  Returns 1 for success.  Returns zero if
 the new protocol fails, but the old protocol still might work if it
 gets tried.
 *********************************************************************
 */
static int
new_protocol_read(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 y_bus,
          LONGSIZ32 addr, LONGSIZ32 *data, int handshake)
{
    SHORTSIZ16 error;
    LONGSIZ32 reg;
    unsigned long vec;
    int     i;

    CHK(i1432_direct_read32_register(mn, E1432_HOSTB_ICS_REG,
                     (LONGSIZ32 *) &reg));
    if ((reg & E1432_HOSTB_ICS_DMAE) != 0)
    {
	/* Use new protocol when the DMAE bit is set in the ICS
	   register.  This should only be the case when the 96002
	   firmware is trying to use DMA mode on the host port, and we
	   don't want to mess that up by flushing valuable data out of
	   the host port.  We can't just use this new protocol all the
	   time, because there is older hardware out there which
	   doesn't have the new protocol in its bootrom. */

	if (y_bus)
	    vec = E1432_CVR_NEW_Y_READ;
	else
	    vec = E1432_CVR_NEW_X_READ;

	CHK(i1432_direct_write32_register(mn, E1432_MEMIO_ADDR, addr));
	if (handshake)
	{
	    CHK(i1432_direct_write32_register(mn, E1432_MEMIO_DATA, addr + 1));
	}
	CHK(i1432_direct_write32_register(mn, E1432_HOSTB_CVR_REG, vec));
	for (i = 0; i < E1432_ICS_WAIT_MAX; i++)
	{
	    CHK(i1432_direct_read32_register(mn, E1432_MEMIO_DATA, data));
	    if (handshake)
	    {
		CHK(i1432_direct_read32_register(mn, E1432_MEMIO_ADDR,
			     (LONGSIZ32 *) &reg));
		if (*data == reg)
		    return 1;
	    }
	    else
		return 1;
	}
    }
    return 0;
}

/*
 *********************************************************************
 Returns -1 for failure.  Returns 0 for success.
 *********************************************************************
 */
static int
old_protocol_read(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 y_bus,
          LONGSIZ32 addr, LONGSIZ32 *data, int handshake)
{
    SHORTSIZ16 error;
    unsigned long reg, mem;

    /* set VXI addresses */
    if (y_bus)
        mem = E1432_HOSTB_Y_READ_REG;
    else
        mem = E1432_HOSTB_X_READ_REG;

    /* write address to Memory Interrupt */
    if (handshake)
    {
	CHK(waitIcsBit(mn, E1432_HOSTB_ICS_TXDE, 1));
    }
    CHK(i1432_direct_write32_register(mn, mem, addr));

    if (handshake)
    {
	/* Check for RX already full */
	CHK(i1432_direct_read32_register(mn, E1432_HOSTB_ICS_REG,
                     (LONGSIZ32 *) &reg));
	if ((reg & (E1432_HOSTB_ICS_HMRC | E1432_HOSTB_ICS_RXDF))
	    == (E1432_HOSTB_ICS_HMRC | E1432_HOSTB_ICS_RXDF))
	{
	    /* HMRC == 1 -> not our response
	       RXDF == 1 -> other data is there
	       So, clear RX by reading it */
	    CHK(i1432_direct_read32_register(mn,
                         E1432_HOSTB_RXTX_REG,
                         (LONGSIZ32 *) &reg));
	}

	CHK(waitIcsBit(mn, E1432_HOSTB_ICS_HMRC, 0));
    }

    /* Read data after Memory Interrupt */
    CHK(i1432_direct_read32_register(mn, E1432_HOSTB_RXTX_REG, data));

    return 0;
}

/*
 *********************************************************************
 Try the new protocol read if appropriate.  If that fails, or if it
 wasn't appropriate, try the old protocol read.  Returns -1 for
 failure.  Returns 0 for success.
 *********************************************************************
 */
static int
smart_read(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 y_bus,
       LONGSIZ32 addr, LONGSIZ32 *data, int handshake)
{
    int     error;

    error = new_protocol_read(mn, y_bus, addr, data, handshake);
    if (error < 0)
	return -1;
    if (error == 0)
    {
	CHK(old_protocol_read(mn, y_bus, addr, data, handshake))
    }
    return 0;
}

/*
 *********************************************************************
 This function reads a word from an address in the 96000 memory space.
 Since the entire 96000 address space is not mapped to VXI, the
 96000 does the memory access via the host port.  So, the 96000
 must be up and running.

 >busBY     0 for 96000 X mem and A bus, 1 for Y mem B bus
 >dspAddr       96000 address
 <dspData   32-bit data read
 returns    0 of ok
 *********************************************************************
 */

SHORTSIZ16 EXPORT
e1432_read_dsp_mem(E1432ID hw,
           SHORTSIZ16 chanID,
           SHORTSIZ16 busBY,
           LONGSIZ32 dspAddr,
           LONGSIZ32 *dspData)
{
#ifndef HAVE_VTL
    E1432_MODULE_LIST_NODE dmn; /* default module list node */
    E1432_MODULE_LIST_NODE *mn;
    char    str[E1432_SICL_NAME_MAX + 2];	/* Allow for ",8" */
    int     error;

    TRACE_PRINTF(0, ("e1432_read_dsp_mem(0x%p, %d, %d, 0x%lx, 0x%p)\n",
             hw, chanID, busBY, dspAddr, dspData));

    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
    {
	/* allow peek/poke to work before assing_channel..  */
	(void) sprintf(str, "%s,8", i1432_sicl_name);
	dmn.sicl_id = iopen(str);
	if (dmn.sicl_id == 0)
	    return i1432_la_print_error(8, ERR1432_UNABLE_TO_OPEN_SICL);

	dmn.a24_base = (LONGSIZ32 *) (void *)
	    imap(dmn.sicl_id, I_MAP_EXTEND, 0, 8, NULL);
	dmn.la = 8;
	dmn.chan_id = 1;
	dmn.nchan = 1;
	mn = &dmn;
    }

    CHK(smart_read(mn, busBY, dspAddr, dspData, 1));

#endif
    return 0;
}

/*
 *********************************************************************
 This function writes a word to byte-wide port in 96000 memory space.
 It is used to write command to the Clarinet 56000 host port.
 Since the entire 96000 address space is not mapped to VXI, the
 96000 does the memory access via the host port.  So, the 96000
 must be up and running.

 >busBY     0 for 96000 X mem and A bus, 1 for Y mem B bus
 >bytesPerWord  number of byte-size writes to do. MSB first.
 >dspAddr       96000 address
 >dspData   32-bit data to write
 returns    0 of ok
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_write_dsp_port(E1432ID hw,
             SHORTSIZ16 chanID,
             SHORTSIZ16 busBY,
             SHORTSIZ16 bytesPerWord,
             LONGSIZ32 dspAddr,
             LONGSIZ32 dspData)
{
    SHORTSIZ16 error;
    E1432_MODULE_LIST_NODE *mn;
    unsigned long mem;      /* vxi address of 96000 port */
    unsigned long temp;
    int     handshake = 1;
    int     j;          /* bytesPerWord-1 ... 0 */

    TRACE_PRINTF(0, ("e1432_write_dsp_port(0x%p, %d, %d, %d, 0x%lx, 0x%lx)\n",
             hw, chanID, busBY, bytesPerWord, dspAddr, dspData));

    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    CHK(i1432_get_module_from_chan(hw, chanID, &mn));

    /* disable handshake on e1406 since it is so slow */
#ifdef EMULATE_SICL
    if ((bytesPerWord>1) && (INTERFACE_E1406 == esicl_interface_type()))
        handshake = 0;
#endif

    if (busBY)
        mem = E1432_HOSTB_Y_WRITE_REG;
    else
        mem = E1432_HOSTB_X_WRITE_REG;

    if (bytesPerWord)
    for (j = bytesPerWord - 1;  j >= 0; j--)
    {
        /* write address to Memory Interrupt */
        if (handshake)
        {
	    CHK(waitIcsBit(mn, E1432_HOSTB_ICS_TXDE, 1));
        }
        CHK(i1432_direct_write32_register(mn, mem, dspAddr - (j * 4)));

        /* write data to Memory Interrupt */
        temp = (dspData >> (j * 8)) & 0x00FFUL;
        temp = (temp << 24L) & 0xFF000000UL; /* move byte to MSB */
        if (handshake)
        {
	    CHK(waitIcsBit(mn, E1432_HOSTB_ICS_TXDE, 1));
        }
        CHK(i1432_direct_write32_register(mn, mem, temp));
    }
    return 0;
}

/*
 *********************************************************************
 This function reads a word from byte-wide port in 96000 memory space.
 It is used to read from Clarinet 56000 host port.
 Since the entire 96000 address space is not mapped to VXI, the
 96000 does the memory access via the host port.  So, the 96000
 must be up and running.

 >busBY     0 for 96000 X mem and A bus, 1 for Y mem B bus
 >bytesPerWord  number of byte-size reads to do. MSB first.
 >dspAddr       96000 address
 <dspData   pointer to 32-bit data
 returns    0 of ok
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_read_dsp_port(E1432ID hw,
            SHORTSIZ16 chanID,
            SHORTSIZ16 busBY,
            SHORTSIZ16 bytesPerWord,
            LONGSIZ32 dspAddr,
            LONGSIZ32 *dspData)
{
    E1432_MODULE_LIST_NODE *mn;
    unsigned long temp;
    int     handshake = 1;
    int     j;          /* bytesPerWord-1 ... 0 */
    int     error;

    TRACE_PRINTF(0, ("e1432_read_dsp_port(0x%p, %d, %d, %d, 0x%lx, 0x%p)\n",
		     hw, chanID, busBY, bytesPerWord, dspAddr, dspData));

    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    CHK(i1432_get_module_from_chan(hw, chanID, &mn));

#ifdef EMULATE_SICL
    /* Disable handshake on e1406 since it is so slow */
    if ((bytesPerWord>1) && (INTERFACE_E1406 == esicl_interface_type()))
        handshake = 0;
#endif

    *dspData = 0;
    if (bytesPerWord)
    for (j = bytesPerWord - 1;  j >= 0; j--)
    {
        CHK(smart_read(mn, busBY, dspAddr - (j * 4),
               (LONGSIZ32 *) &temp, handshake));
        *dspData = ((*dspData) << 8) | ((temp >> 24) & 0xffUL);
    }

    TRACE_PRINTF(0, ("e1432_read_dsp_port got 0x%x)\n", *dspData));
    return 0;
}


/*
 *********************************************************************
 This function reads the firmware marker register.  The firmware marker
 is a 96000 memory location that is written to with different values
 as the firmware boots.  After something like e1432_install() fails,
 it can be read to figure out where it choked.
 
 >la     	logical address
 <marker	returned firmware marker value.
 returns    	0 of ok
 *********************************************************************
 */

SHORTSIZ16 EXPORT
e1432_read_fw_marker( SHORTSIZ16 la, LONGSIZ32 * marker)
{
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    int      save_chan_count;
    SHORTSIZ16 error = 0;
    SHORTSIZ16 error2;

    TRACE_PRINTF(0, ("e1432_read_fw_marker(%d, 0x%p)\n", (int)la, marker));
    *marker = 0;
    
        /* Do this before calling i1432_fake_setup_sicl */
    mn->sicl_id = 0;
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    /* Set up SICL for this module */
    /* Set extra_io to zero; module might be completely hosed */
    error = i1432_fake_setup_sicl(mn, la, &tmp_chan_list, 0);
    if (error)
        goto cleanup;
                                                                        
                                                                        
    error = i1432_direct_read32_register(mn, E1432_FW_MARKER_REG, marker);
    if (error)
    	goto cleanup;

    TRACE_PRINTF(0, ("  FW_MARKER = 0x%08lX\n", *marker));
    	
 cleanup:
    error2 = i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    if (error2)
	error = error2;
    return error;                  
}

